/************************************************************************

CDInterface.hpp

Copyright (C) 2007 Virus
Copyright (C) 2002 mooby

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

************************************************************************/
#pragma warning(disable:4018)

#ifndef CDINTERFACE_HPP
#define CDINTERFACE_HPP

#include "TrackInfo.hpp"
#include "SubchannelData.hpp"
#include "CDDAData.hpp"
#include "TrackParser.hpp"
#include "FileInterface.hpp"
#include "Utils.hpp"

//#include <vorbis/vorbisfile.h>
//#include <glib-2.0/glib.h>

extern TDTNFormat tdtnformat;

// a container class for holding all the CD info needed
class CDInterface
{
    public:
        CDInterface() : scd(NULL), cdda(NULL), image(NULL)
        {
            // based on the program name, decide what order to return the
            // track info.  fyi, tester is my testing/development program.
            std::string programName = getProgramName();
            if ( programName == "epsxe" || programName == "pcsx"  ||
                 programName == "pcsx2" || programName == "tester" )
                tdtnformat = fsmint;
            else
                tdtnformat = msfbcd;
        }

        inline bool hasOggFiles(const std::string& fileName);
        inline void open(const std::string& fileName);

        ~CDInterface()
        {
            if (cdda)  delete cdda;
            if (image) delete image;
            if (scd)   delete scd;
        }

        // returns the number of tracks - 1 because trackList[0] is 
        // the full CD length
        inline unsigned long getNumTracks() const {return trackList.size()-1;}

        // returns the TrackInfo for trackNum
        inline TrackInfo getTrackInfo(const unsigned long trackNum) const
            throw(Exception);

        // seeks the data pointer to time
        inline void moveDataPointer(const CDTime& time) throw(Exception)
        {
            image->seek(time); 
            scd->seek(time);
        }
        inline unsigned char* readDataPointer() const {return image->getBuffer();}
        inline unsigned char* readSubchannelPointer() const {return scd->get();}
        inline CDTime getCDLength() {return image->getCDLength();}

        // starts CDDA playing at time
        inline int playTrack(const CDTime& time) {return cdda->play(time);}
        inline int stopTrack() {return cdda->stop();}
        inline CDTime readTime(void) 
        {
            return cdda->isPlaying() ? cdda->playTime() : image->getSeekTime();
        }
        inline bool isPlaying (void) {return cdda->isPlaying();}

    private:
        // a vector of track info that stores the start, end, and length of each track
        std::vector<TrackInfo> trackList;
        // the subchannel data for the cd
        SubchannelData* scd;
        // the CDDA data for the cd
        CDDAData* cdda;
        // the interface to the CD image itself
        FileInterface* image;
};
/*
inline bool CDInterface::hasOggFiles(const std::string& str)
{
    std::string teststr1 = stripFileExt(str)+".01.ogg";
    cout << "Looking for:" << teststr1 << "\n";
    if (g_file_test(teststr1.c_str(),G_FILE_TEST_IS_REGULAR))
    {
        cout << "File found, using ogg\n";
        return true;
    } 
    return false;
}
*/
// initalizes all the data for CDInterface based on a file name from CDOpen
inline void CDInterface::open(const std::string& fileName)
{
    printf("CDInterface::open(%s)\n",fileName.c_str());
    // use the FIFactory to make the FileInterface
    std::string extension;
    image = FileInterfaceFactory(fileName, extension);

    std::string fileroot = fileName;
    fileroot.erase(fileroot.rfind(extension));

    TrackParser* tp = TrackParserFactory(fileroot, image);
    tp->parse();
    tp->postProcess(image->getCDLength());
    trackList = tp->getTrackList();
    delete tp;

    fprintf(stdout, "Using tracklist:\n");
    for (int i = 0; i < trackList.size(); i++)
        cout << trackList[i];

    // if there is more than one track, initalize the CDDA data
    if (trackList.size() > 2)
    {
     //   if (hasOggFiles(fileName)) // && prefs for using ogg?
     //       cdda = new PlayOGGData(trackList);
       // else
          cdda = new PlayCDDAData(trackList);
        cdda->openFile(fileName);
    }
    else
        cdda = new NoCDDAData();

    // build the subchannel data
    scd = SubchannelDataFactory(fileroot);
}

// returns the TrackInfo for trackNum if it exists
inline TrackInfo CDInterface::getTrackInfo(const unsigned long trackNum) const
throw(Exception)
{
    if (trackNum >= trackList.size())
    {
        std::ostringstream ost;
        ost << trackNum << std::endl;
        Exception e(std::string("Track number out of bounds") + ost.str());
        THROW(e);
    }
    return trackList[trackNum];
}

#endif
